/* ============================================
   app.js - Funciones globales del sistema web
   ============================================ */

/* ----- Utilidades ----- */
const $ = (sel) => document.querySelector(sel);
const $$ = (sel) => document.querySelectorAll(sel);

/* ----- Notificaciones ----- */
function notify(msg, type = "info") {
  let div = document.createElement("div");
  div.className = `alert alert-${type}`;
  div.textContent = msg;
  document.body.prepend(div);
  setTimeout(() => div.remove(), 4000);
}

/* ----- Validaciones ----- */
function validarEmail(email) {
  return /^[^\\s@]+@[^\\s@]+\\.[^\\s@]+$/.test(email);
}

function validarDNI(dni) {
  dni = dni.toUpperCase();
  if (!/^[0-9]{8}[A-Z]$/.test(dni)) return false;
  const letras = "TRWAGMYFPDXBNJZSQVHLCKE";
  const numero = parseInt(dni.substring(0, 8), 10);
  return dni[8] === letras[numero % 23];
}

/* ----- Confirmación antes de enviar formularios ----- */
$$("form").forEach((f) => {
  f.addEventListener("submit", (e) => {
    const emailInput = f.querySelector("input[name='email']");
    const dniInput = f.querySelector("input[name='dni']");
    if (emailInput && !validarEmail(emailInput.value)) {
      e.preventDefault();
      notify("Correo electrónico no válido", "error");
      return;
    }
    if (dniInput && !validarDNI(dniInput.value)) {
      e.preventDefault();
      notify("DNI o NIF no válido", "error");
      return;
    }
    if (!confirm("¿Deseas continuar con esta acción?")) {
      e.preventDefault();
    }
  });
});

/* ----- Alternar modo oscuro/claro ----- */
function toggleDarkMode() {
  document.body.classList.toggle("dark");
  localStorage.setItem(
    "theme",
    document.body.classList.contains("dark") ? "dark" : "light"
  );
}

/* ----- Restaurar modo guardado ----- */
window.addEventListener("DOMContentLoaded", () => {
  if (localStorage.getItem("theme") === "dark") {
    document.body.classList.add("dark");
  }

  // Botón modo oscuro (si existe)
  const darkBtn = $("#btn-darkmode");
  if (darkBtn) darkBtn.addEventListener("click", toggleDarkMode);
});

/* ----- Protección básica contra XSS en inputs visibles ----- */
$$("input[type='text'], textarea").forEach((el) => {
  el.addEventListener("input", (e) => {
    e.target.value = e.target.value.replace(/[<>]/g, "");
  });
});

/* ----- Gestión dinámica de paneles ----- */
if ($(".panel")) {
  // Ejemplo de función que carga contenido vía AJAX (opcional)
  async function cargarServicios() {
    try {
      const res = await fetch("api/api_servicios.php");
      const data = await res.text();
      $("#servicios").innerHTML = data;
    } catch (err) {
      notify("Error cargando servicios", "error");
    }
  }
}

/* ----- Efecto visual simple para botones ----- */
$$("button").forEach((btn) => {
  btn.addEventListener("mousedown", () => btn.classList.add("active"));
  btn.addEventListener("mouseup", () => btn.classList.remove("active"));
});
